﻿using System;
using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.SceneManagement;

public class GameController : MonoBehaviour {
  public Enemy debugEnemy;
  public UIController uiController;

  public AudioSource pickupSound;
  public AudioSource throwSound;
  public AudioSource connectSound;
  public AudioSource endcapSound;

  public Color dayColor;
  public Color nightColor;
  public Color dayCameraBg;
  public Color nightCameraBg;
  public Color daySunBg;
  public Color nightSunBg;
  public float daySunAngle;
  public float nightSunAngle;
  public float dayCycleTime;
  public float nightTransitionTime;
  public Light sun;
  public int campgroundSize;
  public float drainRate;
  public GameObject deathfx;

  public bool AlwaysTryWin { get; set; }

  private List<Enemy> knownEnemies;
  private Player player;
  private float t;
  private float transitionTimer;
  private IEnumerator transitionRoutine;
  private int piecesFound;
  private DateTime dt;
  private float batteryCharge = 0f;

  private static DateTime night = new DateTime(2005, 1, 1, 8, 0, 0);

  private bool IsNight {
    get {
      return transitionTimer > nightTransitionTime / 2f;
    }
  }

  private enum State {
    MENU,
    GAME,
  };
  private State state = State.MENU;

  private void Start() {
    var minutes = dayCycleTime;
    dt = night.AddMinutes(-minutes);

    player = GameObject.FindGameObjectWithTag("Player").GetComponent<Player>();
    knownEnemies = new List<Enemy>();
    if (debugEnemy != null) {
      NotifyEnemySpawned(debugEnemy);
    }
    SyncDayCycle();
    player.gameObject.SetActive(false);
    uiController.UpdateBattery(batteryCharge);
    uiController.UpdateTime(dt);

    // TODO: Difficulty.
    StartGame();
  }

  private void Update() {
    if (state != State.GAME) {
      return;
    }
    t += Time.deltaTime;
    if (t > dayCycleTime && transitionRoutine == null) {
      transitionTimer = 0;
      transitionRoutine = Transition();
      StartCoroutine(transitionRoutine);
    }

    dt = dt.AddMinutes(Time.deltaTime);
    if (IsNight) {
      batteryCharge -= drainRate * Time.deltaTime;
      if (batteryCharge > 0) {
        player.EnableHeadlamp();
      } else {
        player.DisableHeadlamp();
      }
    }

    uiController.UpdateTime(dt);
    uiController.UpdateBattery(batteryCharge);
  }

  public void Death() {
    StartCoroutine(DeathRoutine());
  }

  public void Win() {
    player.gameObject.SetActive(false);
    uiController.ShowWin();
  }

  private IEnumerator DeathRoutine() {
    GameObject.Instantiate(deathfx, player.transform.position, Quaternion.identity);
    player.gameObject.SetActive(false);
    yield return new WaitForSeconds(2f);
    uiController.ShowDeath();
    yield return new WaitForSeconds(2f);

    // TODO: Quick restart
    SceneManager.LoadScene("Game");
  }

  public void StartGameEasy() {
    batteryCharge = 0.5f;
    state = State.GAME;
    player.gameObject.SetActive(true);
  }

  public void StartGame() {
    state = State.GAME;
    player.gameObject.SetActive(true);
  }

  [Ice.ExposeMethod]
  public void SyncDayCycle() {
    RenderSettings.ambientLight = dayColor;
    Camera.main.backgroundColor = dayCameraBg;
    sun.color = daySunBg;
    sun.transform.rotation =
        Quaternion.Euler(daySunAngle, -30, 0);
  }

  public void NotifyPickupSpawned(Pickup p) {

  }

  public void NotifyEnemySpawned(Enemy enemy) {
    knownEnemies.Add(enemy);
    enemy.SetPlayer(player);
  }

  public void NotifyItemThrown(Pickup p) {
    uiController.HidePickup();
    foreach (var enemy in knownEnemies) {
      if (!enemy.gameObject.activeInHierarchy) {
        continue;
      }
      enemy.NotifyThrowable(p);
    }
  }

  public void NotifyItemPickedUp(Pickup p) {
    uiController.ShowBall();
    foreach (var enemy in knownEnemies) {
      if (!enemy.gameObject.activeInHierarchy) {
        continue;
      }
      enemy.NotifyPickupBall(p);
    }
  }

  public void PickupBattery() {
    batteryCharge = Mathf.Min(batteryCharge + 0.3f, 1);
    uiController.UpdateBattery(batteryCharge);
    return;
  }

  public void NotifyPieceActivated(Piece p) {
    piecesFound++;
    var amt = piecesFound / (float)campgroundSize;
    uiController.SetExploredAmount(amt);

    if (amt >= 0.9f) {
      AlwaysTryWin = true;
    }
  }

  public static GameController Find() {
    return GameObject.FindGameObjectWithTag("GameController").GetComponent<GameController>();
  }

  private IEnumerator Transition() {
    while (transitionTimer < nightTransitionTime) {
      transitionTimer += Time.deltaTime;
      var func = EasingFunction.GetEasingFunction(EasingFunction.Ease.Linear);
      float t = func(0, 1, transitionTimer / nightTransitionTime);
      RenderSettings.ambientLight = Color.Lerp(dayColor, nightColor, t);
      Camera.main.backgroundColor = Color.Lerp(dayCameraBg, nightCameraBg, t);
      sun.color = Color.Lerp(daySunBg, nightSunBg, t);
      sun.transform.rotation =
          Quaternion.Euler(Mathf.Lerp(daySunAngle, nightSunAngle, t), -30, 0);
      yield return null;
    }
  }

  public void BarkSound() {

  }
}
