﻿using System.Collections;
using System.Collections.Generic;
using System.Linq;
using UnityEngine;
using Ice.Extensions;
using System;

public class Generator : MonoBehaviour {
  private readonly static System.Random random = new System.Random();

  public Player player;
  public GameObject endcapPrefab;
  public Piece[] pieces;
  public Piece origin;
  public Piece winPiece;
  public PathController pathController;
  public Vector3 spawnPosition = new Vector3(-100, -100, -100);

  [System.Serializable]
  public class SpawnDefinition {
    public bool empty;
    public bool guaranteeFirst;
    public GameObject objectToSpawn;
    public float weight;
    public bool spawnInFinalPosition;
  }
  public SpawnDefinition[] spawnDefinitions;

  private Tuple<float, SpawnDefinition>[] weightedDefinitions;
  private SpawnDefinition firstDef = null;
  private GameController gc;
  private bool spawnedWinPiece;

  // Enable only for debugging because it slows things down.
  public bool debugNaming;

  private Stack<Piece> nextPieces;
  private IEnumerator routine;
  private int pieceCount;

  private void Start() {
    gc = GameController.Find();

    // Generate weighted array.
    var sum = spawnDefinitions.Sum(def => def.weight);
    float acc = 0;
    weightedDefinitions = spawnDefinitions.Select(
        def => {
          acc += def.weight;
          return new Tuple<float, SpawnDefinition>(acc / sum, def);
        }).ToArray();
    firstDef = spawnDefinitions.First(def => def.guaranteeFirst);

    PopulateNextPieces();
    PieceActivated(origin);
  }

  [Ice.ExposeMethod]
  public void DebugGenerateNext() {
    PopulateNextPieces();
    var connection = DebugGetFirstOpenConnection();
    routine = Generate(connection);
  }

  [Ice.ExposeMethod]
  public void DebugStep() {
    if (routine != null) {
      routine.MoveNext();
    }
  }

  [Ice.ExposeMethod]
  public void Cleanup() {
    GameObject[] children = new GameObject[transform.childCount];
    for (int i = 0; i < transform.childCount; i++) {
      children[i] = transform.GetChild(i).gameObject;
    }
    foreach (var child in children) {
      // TODO: Buggy
      if (child.name != "Origin") {
        DestroyImmediate(child);
      }
    }
  }

  // A piece is visible and beginning to activate, prepare its data.
  public void PieceStartActivating(Piece piece) {
    if (piece.SpawnPoint == null) {
      return;
    }

    var selection = SelectSpawnDefinition();

    // If the definition is not empty and the piece has a spawn point, spawn it.
    Debug.Log($"Selected {selection.objectToSpawn.name}");
    if (selection != null && !selection.empty) {
      var spawnPosition = piece.SpawnPoint.transform.position;
      if (selection.spawnInFinalPosition) {
        spawnPosition += Vector3.up * Piece.pieceHiddenDepth;
      }
      var obj = Instantiate(
          selection.objectToSpawn, spawnPosition, Quaternion.identity, piece.transform);
      var enemy = obj.GetComponent<Enemy>();
      if (enemy != null) {
        gc.NotifyEnemySpawned(enemy);
      }
    }
  }

  // Marks a piece as activated and ready to move to.
  // Prepare all outgoing connections and potentially recalculate the path controller.
  public void PieceActivated(Piece piece) {
    StartCoroutine(GenerateAllConnectionsRoutine(piece));
    pathController.PieceActivated(piece);
    gc.NotifyPieceActivated(piece);
  }

  private SpawnDefinition SelectSpawnDefinition() {
    if (firstDef != null) {
      var tmp = firstDef;
      firstDef = null;
      return tmp;
    }

    // Select a random spawn definition by weight.
    // NOTE: Binary search is faster here but this is one of those cases where linear is likely more
    // CPU/MEM efficient for very small values of N like we are dealing with (n < 10).  And easier
    // to implement and debug.  Cool!
    var a = random.NextDouble();
    foreach (var def in weightedDefinitions) {
      if (a <= def.Item1) {
        return def.Item2;
      }
    }

    // Fallback to null.
    return null;
  }

  private IEnumerator GenerateAllConnectionsRoutine(Piece piece) {
    foreach (var c in piece.GetOpenConnections()) {
      PopulateNextPieces();
      yield return StartCoroutine(Generate(c));
    }
  }

  private IEnumerator Generate(Connection connection) {
    var ok = false;
    var wasWinPiece = false;
    while (!ok && nextPieces.Count > 0) {
      var piece = SpawnRandomPiece(out wasWinPiece);
      foreach (var newConnection in piece.GetOpenConnections().Shuffled()) {
        // Align the connections.
        var ofs = VectorUtil.ForwardOffset(-connection.transform.forward) -
                  VectorUtil.ForwardOffset(newConnection.transform.forward);
        piece.transform.Rotate(Vector3.up, ofs);
        var posOfs = connection.transform.position - newConnection.transform.position;
        var newPosition = piece.transform.position + posOfs;
        piece.transform.position = newPosition;

        // NOTE: This is the bug in the dungeon generation setup. If we're modifying positions of
        // colliders in a coroutine, we need to signal to Unity to wait for a fixed timestep before
        // continuing, to ensure colliders are in the right position!
        yield return new WaitForFixedUpdate();

        // Check for collisions.
        //Debug.Log($"Trying {newConnection.gameObject.name}");
        if (!piece.IsColliding(newPosition, connection.transform.parent.gameObject)) {
          connection.MarkConnected(newConnection);
          ok = true;
          break;
        }
      }

      if (ok) {
        // The piece is valid, move it down.
        piece.PushDown();

        // If its the win piece, dont spawn one again.
        if (wasWinPiece) {
          spawnedWinPiece = true;
        }

        break;
      } else {
        // The piece doesn't fit, destroy it and try another.
        //Debug.Log("Piece doesn't fit, trying another.");
        Destroy(piece.gameObject);
      }
    }

    if (!ok) {
      //Debug.Log("No pieces fit!  Marking deadend.");
      connection.DeadEnd = true;
      connection.MarkDeadEnd(endcapPrefab);
    }
  }

  private Connection DebugGetFirstOpenConnection() {
    // Find the first unconnected connection.
    var connections = GetComponentsInChildren<Connection>();
    foreach (var c in connections) {
      if (!c.Connected) {
        return c;
      }
    }
    return null;
  }

  private void PopulateNextPieces() {
    nextPieces = new Stack<Piece>(pieces.Shuffled());
    if (gc.AlwaysTryWin && !spawnedWinPiece) {
      nextPieces.Push(winPiece);
    }
  }

  private Piece SpawnRandomPiece(out bool wasWinPiece) {
    if (nextPieces.Count < 1) {
      Debug.LogError("Pieces exhausted!");
    }
    var prefab = nextPieces.Pop();
    var piece = Instantiate(prefab, spawnPosition, prefab.transform.rotation, transform);
    piece.gameObject.name = $"World Piece #{++pieceCount} ({prefab.name})";
    piece.gameObject.layer = LayerMask.NameToLayer("Piece");
    piece.Hide();
    piece.Gen = this;

    if (debugNaming) {
      Ice.ObjectUtil.AppendNameToChildren(piece.gameObject, $" (Piece {pieceCount})");
    }

    wasWinPiece = prefab == winPiece;

    return piece;
  }
}
