﻿using Pathfinding;
using System.Collections;
using System.Collections.Generic;
using UnityEngine;

[RequireComponent(typeof(AstarPath))]
public class PathController : MonoBehaviour {
  public Transform player;
  public float relocateTolerance = 0.75f;

  private AstarPath astarPath;
  private Pathfinding.GridGraph graph;
  private Bounds toleranceBounds;

  private void Start() {
    astarPath = GetComponent<AstarPath>();
    graph = astarPath.data.gridGraph;
    RecalculateToleranceBounds();
  }

  private void Update() {
    // Check player bounds.
    if (!toleranceBounds.Contains(player.position)) {
      MoveGraph(player.position);
      RecalculateToleranceBounds();
    }
  }

  public void PieceActivated(Piece piece) {
    if (astarPath == null) {
      return;
    }
    astarPath.Scan(graph);
  }

  private void MoveGraph(Vector3 center) {
    // Snap y to a fixed position so rays always work.
    center.y = -0.5f;
    graph.RelocateNodes(center, Quaternion.identity, graph.nodeSize);
    astarPath.Scan(graph);
  }

  private void RecalculateToleranceBounds() {
    var size = new Vector3(graph.Width * relocateTolerance / 2, 10f, graph.Depth * relocateTolerance / 2);
    toleranceBounds = new Bounds(graph.center, size);
  }

  private void OnDrawGizmosSelected() {
    Gizmos.DrawWireCube(toleranceBounds.center, toleranceBounds.size);
  }
}