﻿using System.Linq;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.ProBuilder;
using System.Collections;
using UnityEditor;

public class Piece : MonoBehaviour {
  public static float pieceHiddenDepth = 3f;
  private static float activationSpeed = 0.6f;

  [SerializeField]
  private GameObject[] hiddenObstacles;

  [SerializeField]
  private GameObject _spawnPoint;
  public GameObject SpawnPoint {
    get { return _spawnPoint; }
    set { _spawnPoint = value; }
  }

  public Generator Gen { get; set; }

  [SerializeField]
  private PieceType _type;
  public PieceType Type {
    get { return _type; }
    set { _type = value; }
  }

  public int CollisionMask {
    get {
      if (_collisionMask == -1) {
        _collisionMask = LayerMask.GetMask("Piece") |
                         LayerMask.GetMask("PieceHidden") |
                         LayerMask.GetMask("PieceWall");
      }
      return _collisionMask;
    }
  }
  private int _collisionMask = -1;

  private Connection[] connections;
  private float activationTimer;
  private int hideObstaclesLayer, showObstaclesLayer;

  public enum PieceType {
    TODO,
  }

  private void Start() {
    connections = GetComponentsInChildren<Connection>();
    hideObstaclesLayer = LayerMask.NameToLayer("ObstaclesHidden");
    showObstaclesLayer = LayerMask.NameToLayer("Obstacles");
  }

  #if UNITY_EDITOR
  [Ice.ExposeMethod]
  public void CollectObjects() {
    // Obstacles.
    var layer = LayerMask.NameToLayer("ObstaclesHidden");
    var children = gameObject.GetComponentsInChildren<Transform>();
    hiddenObstacles = children
      .Select(c => c.gameObject)
      .Where(g => g.layer == layer).ToArray();

    // Spawn point.
    var spawn = GetComponentInChildren<SpawnPoint>();
    if (spawn != null) {
      _spawnPoint = spawn.gameObject;
    }

    // Ensure prefab/scene is saved.
    EditorUtility.SetDirty(this);
  }
  #endif

  public void Activate() {
    Show();
    GameController.Find().connectSound.Play();
    activationTimer = 0f;
    Gen.PieceStartActivating(this);
    StartCoroutine(ActivateRoutine());
  }

  public void PushDown() {
    transform.Translate(Vector3.down * pieceHiddenDepth);
  }

  public void Show() {
    gameObject.layer = LayerMask.NameToLayer("Piece");
    foreach (var o in hiddenObstacles) {
      o.gameObject.layer = showObstaclesLayer;
    }
  }

  public void Hide() {
    gameObject.layer = LayerMask.NameToLayer("PieceHidden");
  }

  // Check if this piece collides with pieces other than the one its being connected to
  public bool IsColliding(Vector3 newPosition, GameObject connectingTo) {
    //var myCollider = GetComponentInChildren<Collider>();
    var myCollider = GetComponent<Collider>();
    var boundsOffset = myCollider.bounds.center - transform.position;
    var colliders = Physics.OverlapSphere(
        newPosition + boundsOffset, myCollider.bounds.extents.magnitude, CollisionMask);
    foreach (var c in colliders) {
      var piece = c.GetComponentInParent<Piece>();

      // TODO: Profile this, might suck.
      if (c.gameObject != gameObject &&
          c.gameObject != connectingTo &&
          piece != this &&
          piece.gameObject != connectingTo) {
        //Debug.Log($"Collision between {gameObject.name} and {c.gameObject.name}, ignored is {connectingTo.name}");
        //Debug.Break();
        return true;
      }
    }
    return false;
  }

  [Ice.ExposeMethod]
  public void ListCollisions() {
    var myCollider = GetComponent<Collider>();
    var colliders = Physics.OverlapSphere(
        myCollider.bounds.center, myCollider.bounds.extents.magnitude,
        LayerMask.GetMask("Piece"));
    foreach (var c in colliders) {
      Debug.Log(c.gameObject.name);
    }
  }

  [Ice.ExposeMethod]
  public void CreatePerimeter() {
    var sides = GetSides();
    foreach (var face in sides) {
      var obj = new GameObject();
      obj.transform.parent = transform;
      obj.name = "Perimeter";
      var perimeter = obj.AddComponent<Perimeter>();
      perimeter.Align(GetComponent<ProBuilderMesh>(), face);
    }
  }

  [Ice.ExposeMethod]
  public void CleanupPerimeter() {
    Ice.ObjectUtil.DestroyChildrenImmediate(
        gameObject, child => child.GetComponent<Perimeter>() != null);
  }

  public List<Connection> GetOpenConnections() {
    if (connections == null) {
      connections = GetComponentsInChildren<Connection>();
    }
    return connections.Where(c => !c.Connected && !c.DeadEnd).ToList();
  }

  private IEnumerator ActivateRoutine() {
    // TODO: Make this visual only.
    var start = transform.position;
    var end = transform.position + Vector3.up * pieceHiddenDepth;
    while (activationTimer < activationSpeed) {
      activationTimer += Time.deltaTime;
      float t = EasingFunction.EaseOutQuart(0, 1, activationTimer / activationSpeed);
      transform.position = Vector3.Lerp(start, end, t);
      yield return null;
    }
    transform.position = end;

    // The piece is in place, now prepare its outgoing connections.
    Gen.PieceActivated(this);
  }

  private IList<Face> GetSides() {
    ProBuilderMesh mesh = GetComponent<ProBuilderMesh>();
    var faces = new List<Face>();
    foreach (var face in mesh.faces) {
      Vector3 normal = Math.Normal(mesh, face);
      float dot = Vector3.Dot(normal, Vector3.up);
      if (dot == 0) {
        faces.Add(face);
      }
    }
    return faces;
  }

  //private void OnDrawGizmosSelected() {
  //  var collider = GetComponent<Collider>();
  //  Gizmos.DrawWireSphere(collider.bounds.center, collider.bounds.extents.magnitude);
  //}
}
