﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;

public class Player : MonoBehaviour {

  public bool killable = true;
  public GameObject headlampLight;

  [System.Serializable]
  public class PickupDefinition {
    public Pickup.Type type;
    public GameObject prefab;
  }

  public PickupDefinition[] pickupDefinitions;
  public float maxThrowDist;
  public Pickup.Type pickup;
  public GameObject throwIndicator;

  private Vector3 throwPosition = Vector3.zero;
  private GameController gc;

  public bool HasPickup {
    get {
      return pickup != Pickup.Type.EMPTY;
    }
  }

  public bool HasThrowable {
    get {
      return pickup == Pickup.Type.BALL;
    }
  }

  public Vector3 footPosition {
    get {
      return transform.position + Vector3.down * 1f;
    }
  }

  private void Start() {
    gc = GameController.Find();
  }

  private void Update() {
    // Calculate the throw position.
    Ray ray = Camera.main.ScreenPointToRay(Input.mousePosition);
    var plane = new Plane(Vector3.up, footPosition);
    float enter;
    if (plane.Raycast(ray, out enter)) {
      Vector3 hitPoint = ray.GetPoint(enter);
      var v = Vector3.ClampMagnitude(hitPoint - footPosition, maxThrowDist);
      throwPosition = footPosition + v;
    }

    if (HasThrowable) {
      throwIndicator.SetActive(true);
      if (throwIndicator != null) {
        throwIndicator.transform.position = throwPosition;
      }
      if (Input.GetMouseButtonDown(0)) {
        Throw();
      }
    } else {
      throwIndicator.SetActive(false);
    }
  }

  public void EnableHeadlamp() {
    headlampLight.SetActive(true);
  }

  public void DisableHeadlamp() {

  }

  public bool TryPickupItem(Pickup p) {
    if (p.type == Pickup.Type.BATTERY) {
      gc.PickupBattery();
      gc.pickupSound.Play();
      return true;
    }

    if (pickup != Pickup.Type.EMPTY) {
      return false;
    }
    
    gc.NotifyItemPickedUp(p);
    pickup = p.type;
    gc.pickupSound.Play();
    return true;
  }

  private void Throw() {
    GameObject prefab = null;
    foreach (var def in pickupDefinitions) {
      if (def.type == pickup) {
        prefab = def.prefab;
        break;
      }
    }

    pickup = Pickup.Type.EMPTY;
    if (prefab == null) {
      return;
    }

    var dir = throwPosition - footPosition;
    var obj = Instantiate(prefab, transform.position + transform.up * 0.25f, Quaternion.identity);
    var rb = obj.GetComponent<Rigidbody>();
    if (rb != null) {
      rb.velocity = dir * 2f;
      rb.angularVelocity = new Vector3(
          Random.Range(-1, 10),
          Random.Range(-1, 10),
          Random.Range(-1, 10));
      gc.throwSound.Play();
    }

    // Notify GC about ball to inform enemies.
    gc.NotifyItemThrown(obj.GetComponent<Pickup>());
  }

  private void Die() {
    if (killable)
    gc.Death();
  }

  private void OnTriggerEnter(Collider other) {
    var connection = other.GetComponent<Connection>();

    if (connection != null) {
      connection.ActivateConnection();
      return;
    }

    // Check for win;
    var w = other.GetComponent<Win>();
    if (w != null) {
      gc.Win();
      return;
    }

    // Check for enemy;
    var e = other.GetComponent<Enemy>();
    if (e != null) {
      Die();
      return;
    }

    // Check for pickup;
    var p = other.GetComponent<Pickup>();
    if (p != null) {
      Debug.Log($"Trying to pickup, mine is {pickup}");
      var ok = TryPickupItem(p);
      if (ok) {
        Destroy(p.gameObject);
      }
      return;
    }
  }
}
