﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;

[RequireComponent(typeof(PlayerInput))]
[RequireComponent(typeof(CharacterController))]
public class PlayerMoveController : MonoBehaviour {
  public float gravityFactor = 1f;
  public float groundingForce = 10;
  public float speed = 1f;

  // TODO: This is hack, need to fix this if camera orientation ever changes.
  public float inputRotation = 45;

  private PlayerInput playerInput;
  private CharacterController characterController;
  private Animator animator;
  private AudioSource source;
  private CollisionFlags collisionFlags;
  private Vector3 moveDir;
  private bool groundedLastFrame;

  private void Start() {
    playerInput = GetComponent<PlayerInput>();
    characterController = GetComponent<CharacterController>();
    animator = GetComponentInChildren<Animator>();
    moveDir = Vector3.zero;
    source = GetComponent<AudioSource>();
  }

  private void Update() {
    if (characterController.isGrounded && !groundedLastFrame) {
      moveDir.y = 0f;
    }
    groundedLastFrame = characterController.isGrounded;
    MovePlayer(playerInput.Inputs);
  }

  private void MovePlayer(PlayerInput.InputData inputs) {
    // Build the correct world vector from inputs.
    var worldInput = new Vector3(inputs.direction.x, 0, inputs.direction.y);
    worldInput = Quaternion.AngleAxis(inputRotation, Vector3.up) * worldInput;

    // Project the world movement vector onto the collision plane.
    RaycastHit hit;
    Physics.SphereCast(
        transform.position, characterController.radius, Vector3.down, out hit,
        characterController.height / 2f, Physics.AllLayers, QueryTriggerInteraction.Ignore);
    var projection = Vector3.ProjectOnPlane(worldInput, hit.normal).normalized;

    // Apply to accumulated movement vector.
    moveDir.x = projection.x * speed;
    moveDir.z = projection.z * speed;

    // Update rotation
    if (worldInput.sqrMagnitude > 0) {
      transform.rotation = Quaternion.LookRotation(new Vector3(moveDir.x, 0, moveDir.z));
    }

    // Handle gravity
    if (characterController.isGrounded) {
      // TODO: Jumps here if needed.
      moveDir.y = -groundingForce;
    } else {
      moveDir += Physics.gravity * gravityFactor * Time.deltaTime;
    }

    // Move the character controller.
    collisionFlags = characterController.Move(moveDir * Time.deltaTime);
    var animSpeed = worldInput.sqrMagnitude > 0 ? speed / 3f : 0f;
    if (animSpeed > 0) {
      source.enabled = true;
    } else {
      source.enabled = false;
    }
    animator.SetFloat("speed", animSpeed);
  }
}
