require 'animation'

Enemy = Entity:extend()

function Enemy:init(prop)
    Entity.init(self, prop)
    self.w, self.h = 16, 16
    self.isEnemy = true
end

function Enemy:applyScale(rect, sx, sy)
    if self.orientation == 1 then
        rect.right = rect.left + rect:h() * sy
        rect.top = rect.top + (1 - sy) * rect:w() / 2
        rect.bottom = rect.bottom - (1 - sy) * rect:w() / 2
    elseif self.orientation == 2 then
        rect.bottom = rect.top + rect:h() * sy
        rect.left = rect.left + (1 - sx) * rect:w() / 2
        rect.right = rect.right - (1 - sx) * rect:w() / 2
    elseif self.orientation == 3 then
        rect.left = rect.right - rect:h() * sy
        rect.top = rect.top + (1 - sy) * rect:w() / 2
        rect.bottom = rect.bottom - (1 - sy) * rect:w() / 2
    else
        rect.top = rect.bottom - rect:h() * sy
        rect.left = rect.left + (1 - sx) * rect:w() / 2
        rect.right = rect.right - (1 - sx) * rect:w() / 2
    end
    return rect
end


Walker = Enemy:extend()

function Walker:init(prop)
    Enemy.init(self, prop)
end

function Walker:snap(dir)
    local mx, my = 0, 0
    -- if dir == 'down' and self.left then mx = 1 end
    -- if dir == 'down' and not self.left then mx = -1 end
    -- if dir == 'up' and not self.left then mx = -1 end
    -- if dir == 'up' and self.left then mx = 1 end
    -- if dir == 'left' and self.left then my = 0.1 end
    -- if dir == 'left' and not self.left then my = -1 end
    -- if dir == 'right' and not self.left then my = -1 end
    -- if dir == 'right' and self.left then my = 1 end

    local tile, space = game.currentPlanet:getSolidTileDir(self:cx()+mx*self.w/2, self:cy()+my*self.y/2, dir)
    if tile then
        local tilerect = tile.rect
        if dir == 'down' then
            self.y = tilerect.top - self.h
        elseif dir == 'right' then
            self.x = tilerect.left - self.w
        elseif dir == 'up' then
            self.y = tilerect.bottom
        elseif dir == 'left' then
            self.x = tilerect.right
        end
        return tile, space
    end
    return nil, space
end

function Walker:update(dt)
    local gravdir = DIRMAP[self.orientation]
    local tile_under, space = self:snap(gravdir)
    local walk_vector = {0, 0}
    local adj = 10
    if gravdir == 'down' and self.left == true then 
        walk_vector = {-1, 0}
        if not tile_under then
            self:translate(-adj, adj)
            self:rotate(-1)
            return self:snap()
        else
            local forwardTile = game.currentPlanet:getSolidTileDir(self.x+self.w, self:cy(), 'left')
            if forwardTile then
            -- self:translate(-adj/2, -adj/2)
                self:rotate(1)
            end
        end
    elseif gravdir == 'down' and self.left == false then 
        walk_vector = {1, 0}
        if not tile_under then
            self:translate(adj, adj)
            self:rotate(1)
            return self:snap()
        else
            local forwardTile = game.currentPlanet:getSolidTileDir(self.x, self:cy(), 'right')
            if forwardTile then
            -- self:translate(adj/2, -adj/2)
                self:rotate(-1)
            end
        end
    elseif gravdir == 'right' and self.left == true then 
        walk_vector = {0, 1}
        if not tile_under then
            self:translate(adj, adj)
            self:rotate(-1)
            return self:snap()
        else
            local forwardTile = game.currentPlanet:getSolidTileDir(self:cx(), self.y, 'down')
            if forwardTile then
            -- self:translate(-adj/2, adj/2)
                self:rotate(1)
            end
        end
    elseif gravdir == 'right' and self.left == false then 
        walk_vector = {0, -1}
        if not tile_under then
            self:translate(adj, -adj)
            self:rotate(1)
            return self:snap()
        else
            local forwardTile = game.currentPlanet:getSolidTileDir(self:cx(), self.y+self.h, 'up')
            if forwardTile then
            -- self:translate(-adj/2, -adj/2)
                self:rotate(-1)
            end
        end
    elseif gravdir == 'up' and self.left == true then 
        walk_vector = {1, 0}
        if not tile_under then
            self:translate(adj, -adj)
            self:rotate(-1)
            return self:snap()
        else
            local forwardTile = game.currentPlanet:getSolidTileDir(self.x, self:cy(), 'right')
            if forwardTile then
            -- self:translate(adj/2, adj/2)
                self:rotate(1)
            end
        end
    elseif gravdir == 'up' and self.left == false then 
        walk_vector = {-1, 0}
        if not tile_under then
            self:translate(-adj, -adj)
            self:rotate(1)
            return self:snap()
        else
            local forwardTile = game.currentPlanet:getSolidTileDir(self.x+self.w, self:cy(), 'left')
            if forwardTile then
            -- self:translate(-adj/2, adj/2)
                self:rotate(-1)
            end
        end
    elseif gravdir == 'left' and self.left == true then 
        walk_vector = {0, -1}
        if not tile_under then
            self:translate(-adj, -adj)
            self:rotate(-1)
            return self:snap()
        else
            local forwardTile = game.currentPlanet:getSolidTileDir(self:cx(), self.y+self.h, 'up')
            if forwardTile then
            -- self:translate(adj/2, -adj/2)
                self:rotate(1)
            end
        end
    elseif gravdir == 'left' and self.left == false then 
        walk_vector = {0, 1}
        if not tile_under then
            self:translate(-adj, adj)
            self:rotate(1)
            return self:snap()
        else
            local forwardTile = game.currentPlanet:getSolidTileDir(self:cx(), self.y, 'down')
            if forwardTile then
            -- self:translate(adj/2, adj/2)
                self:rotate(-1)
            end
        end
    end

    self.x = self.x + walk_vector[1] * self.speed * dt
    self.y = self.y + walk_vector[2] * self.speed * dt
end

Boulder = Walker:extend()

function Boulder:init(prop)
    Walker.init(self, prop)
    self.speed = math.random(50, 75)
    self.sprite = Animation(assets.gfx.enemy2, 16, 5 / self.speed)
end

Critter = Walker:extend()

function Critter:init(prop)
    Walker.init(self, prop)
    self.speed = math.random(15, 30)
    self.sprite = Animation(assets.gfx.enemy1, 16, 1 / self.speed)
end

function Critter:update(dt)
    Walker.update(self, dt)
end

function Critter:fixCollision(rect)
    return self:applyScale(rect, 0.75, 0.25)
end


SuperCritter = Critter:extend()

function SuperCritter:init(prop)
    Critter.init(self, prop)
    self.stop = false
    self.speed = math.random(50, 100)
    self.sprite = Animation(assets.gfx.enemy3, 16, 1.5 / self.speed)
    self:resetTimer()
end

function SuperCritter:resetTimer()
    self.timer = math.random(2, 6)
end

function SuperCritter:update(dt)
    if not self.stop then
        Critter.update(self, dt)
    end

    self.timer = self.timer - dt
    if self.timer < 0 then
        self:resetTimer()
        local rand = math.random()
        if rand < 0.45 then
            self.left = true
            self.stop = false
            self.sprite.speed = 1.5 / self.speed
        elseif rand < 0.9 then
            self.left = false
            self.stop = false
            self.sprite.speed = 1.5 / self.speed
        else
            self.stop = true
            self.sprite.speed = 1
        end
    end
end

Spike = Enemy:extend()

function Spike:init(prop)
    Enemy.init(self, prop)
    self.w, self.h = 20, 20
    self.sprite = Animation(assets.gfx.spike, 20, 1/30)
end

function Spike:fixCollision(rect)
    local rect = self:applyScale(rect, 0.25, 0.5)
    return rect
end

function Spike:getTriggerRect()
    local rect = self:getCollisionRect()
    rect.left = rect.left - 12
    rect.top = rect.top - 12
    rect.right = rect.right + 12
    rect.bottom = rect.bottom + 12
    return rect
end

function Spike:update(dt)
    if self:getTriggerRect():intersects(player:getCollisionRect()) then
        self.sprite:forwardOnce()
    else
        self.sprite:reverseOnce()
    end
    Enemy.update(self, dt)
end

Bullet = Enemy:extend()
Bullet.SPEED = 210


function Bullet:init(prop)
    Enemy.init(self, prop)
    self.w, self.h = 8, 8
    self.sprite = Sprite(assets.gfx.bullet)
    self.speed = -Bullet.SPEED
end

function Bullet:update(dt)
    self.x = self.x + self.speed * dt
end


Guard = Enemy:extend()

Guard.State = {
    idle=0,
    walking=1,
    firing=2,
}

function Guard:init(prop)
    Enemy.init(self, prop)
    self.w, self.h = 20, 20
    self.timer = {
        decision = 1,
        firing = 0,
    }
    self.speed = math.random(8, 15)
    self.sprites = {
        idle = Sprite(assets.gfx.enemy4_idle),
        fire = Sprite(assets.gfx.enemy4_fire),
        run = Animation(assets.gfx.enemy4_run, 20, 1/self.speed),
    }
    self.state = Guard.State.idle
    self.sprite = self.sprites.idle
    self.walk_amount = 0
    self.walk_bounds = 20
end

function Guard:idle()
    self.state = Guard.State.idle
    self.sprite = self.sprites.idle
end

function Guard:fire()
    self.state = Guard.State.firing
    self.sprite = self.sprites.fire
end

function Guard:randomWalk()
    self.state = Guard.State.walking
    self.sprite = self.sprites.run
    if math.random() > 0.5 then self.left = true else self.left = false end
end

function Guard:decide()
    -- Check if we should fire
    if self.orientation == 0 and math.abs(self.x - player.x) <= 130 then
        if self.left and player.x <= self.x then
            return self:fire()
        elseif not self.left and player.x >= self.x then
            return self:fire()
        end
    end

    if self.state == Guard.State.idle then
        return self:randomWalk()
    end

    if math.random() > 1 / 3 then self:randomWalk() else self:idle() end
end

function Guard:update(dt)
    Entity.update(self, dt)

    for k, timer in pairs(self.timer) do
        self.timer[k] = self.timer[k] - dt
    end

    if self.timer.decision < 0 then
        self:decide()
        self.timer.decision = math.random() * 12 / self.speed + 0.5
    end

    if self.state == Guard.State.walking then
        local move_vector = Vector(0,0)
        if self.orientation == 0 then move_vector.x = -1
        elseif self.orientation == 1 then move_vector.y = -1
        elseif self.orientation == 2 then move_vector.x = 1
        elseif self.orientation == 3 then move_vector.y = 1 end
        if not self.left then
            move_vector = move_vector * -1
        end
        move_vector = move_vector * dt * self.speed
        local len = math.max(math.abs(move_vector.x), math.abs(move_vector.y))
        if self.left then
            self.walk_amount = self.walk_amount - len
            if self.walk_amount < -self.walk_bounds then
                self.walk_amount = -self.walk_bounds
                self:decide()
            else
                self.x = self.x + move_vector.x
                self.y = self.y + move_vector.y
            end
        else
            self.walk_amount = self.walk_amount + len
            if self.walk_amount > self.walk_bounds then
                self.walk_amount = self.walk_bounds
                self:decide()
            else
                self.x = self.x + move_vector.x
                self.y = self.y + move_vector.y
            end
        end
    elseif self.state == Guard.State.firing then
        if self.timer.firing < 0 then
            self.timer.firing = 15 / self.speed
            game.fireBullet(self:cx(), self:cy() + 2, self.left)
        end
    end
end

function Guard:fixCollision(rect)
    local rect = self:applyScale(rect, 0.25, 0.5)
    return rect
end

function Guard:draw()
    Entity.draw(self)
    if self.debug then
        love.graphics.setLineWidth(2)
        love.graphics.setColor(255, 0, 0)
        love.graphics.line(0, 0, self.debug.x, self.debug.y)
    end
end
