game = Context()

-- Settings
game.SCALE = 2
game.COLLISION = false
game.INVINCIBLE = false

-- Global functions
function game.getWidth()
    return love.graphics.getWidth() / game.SCALE
end

function game.getHeight()
    return love.graphics.getHeight() / game.SCALE
end

function game.toScreen(wx, wy)
    return wx * game.SCALE, wy * game.SCALE
end

require 'entity'
require 'player'
require 'planet'
require 'dialog'
require 'level'
require 'menu'
require 'particles'

-- Objects
game.entities = List()
game.currentLevel = nil
game.currentPlanet = nil


-- Menus
game.deathMenu = Menu("You're dead.")
game.deathMenu:push("Respawn", function() return game.reload() end)
game.deathMenu:push("Quit", function() return game.exit() end)
game.activeMenu = nil


-- Graphic stuff
game.overlay = 0.0
game.shaderMod = 0.0
game.activeShader = nil

function game.load()
    game.timeleft = 15 * 60
    game.hideClock = true
    game.hideClockTimer = 0.6

    -- Setup runtime objects
    particles:init()

    -- Setup player
    player:init()
    game.entities:insert(player)
    game.entities:insert(player.friend)
    game.resetPlayer()
    console:write("Game initialized")

    -- Load first level
    game.levelCount = STARTLEVEL
    game.loadLevel(game.levelCount)
end

function game.update(dt)
    -- Globals
    time.update(dt)
    tween.update(dt)
    dialog:update(dt)
    particles:update(dt)
    game.currentPlanet:updateEffects(dt)

    -- Shaders
    if game.activeShader then
        game.activeShader:send('mod', game.shaderMod)
    end

    -- Physics engine
    if not dialog.active and not game.activeMenu then
        game.currentPlanet:update(dt)

        -- Update objects
        game.timeleft = game.timeleft - dt
        if game.timeleft <= 0 and not game.completed then
            game.lose()
        end
        game.hideClockTimer = game.hideClockTimer - dt
        if game.hideClockTimer < 0 and not game.completed then
            game.hideClock = false
        end
        for entity in game.entities:iter() do
            if isinstance(entity, Bullet) then
                if entity.x < 0 or entity.x > game.getWidth() or entity.y < 0 or entity.y > game.getHeight() then
                    game.entities:remove(entity)
                else
                    entity:update(dt)
                end
            else
                entity:update(dt)
            end
        end

        -- Check player bounds
        if not player.dead and player.y > game.getHeight() then
            game.die()
        end
    end
end

function game.getClockText()
    local minutes = math.floor(game.timeleft / 60)
    local seconds = math.floor(game.timeleft % 60)
    local text = minutes .. ":"
    if seconds < 10 then
        text = text + "0"
    end
    text = text + seconds
    return text
end

function game.drawClock()
    love.graphics.setColor(255, 255, 255)
    love.graphics.setFont(assets.font.large)
    love.graphics.printf(game.getClockText(), 0, 20, love.graphics.getWidth(), 'center')
end

function game.draw()
    love.graphics.setColor(255, 255, 255)

    -- Draw everything inside scale
    love.graphics.push()
        love.graphics.scale(game.SCALE, game.SCALE)

        game.currentPlanet:drawBg()

        -- Apply filter
        love.graphics.setPixelEffect(game.activeShader)

        -- Draw world
        game.currentPlanet:draw()

        -- Draw objects
        for entity in game.entities:iter() do
            entity:transformDraw()
        end

        -- Draw extra things in world coords
        if dialog.active then
            love.graphics.draw(assets.gfx.speech, player.friend.x - 10, player.friend.y - 10)
        end
        love.graphics.setPixelEffect()
    love.graphics.pop()

    -- Draw effects
    particles:draw()

    -- Draw text HUDs
    dialog:draw()
    if game.activeMenu then
        game.activeMenu:draw()
    end

    if not game.hideClock then
        game.drawClock()
    end

    -- Draw fullscreen overlays
    love.graphics.setColor(255, 255, 255, game.overlay * 255)
    love.graphics.rectangle('fill', 0, 0, love.graphics.getWidth(), love.graphics.getHeight())

    -- Draw debug stuff
    if DEBUG then
        love.graphics.setColor(255, 255, 0)
        love.graphics.setFont(assets.font.small)
        love.graphics.print(love.timer.getFPS(), love.graphics.getWidth() - 40, 10)
        console:draw()
    end
end

function game.keypressed(key, unicode)
    if key == 'f11' then
        DEBUG = not DEBUG
        return
    end

    if game.activeMenu then
        game.activeMenu:keypressed(key, unicode)
    else
        if not dialog:keypressed(key, unicode) then
            player:keypressed(key, unicode)
        end

        if DEBUG then
            if key == 'f2' then
                game.reload()
            elseif key == 'f3' then
                game.currentPlanet:rotatePlanet(-1)
            elseif key == 'f4' then
                game.COLLISION = not game.COLLISION
            elseif key == 'f5' then
                game.INVINCIBLE  = not game.INVINCIBLE
            elseif key == 'pageup' then
                game.nextLevel()
            elseif key == 'pagedown' then
                game.prevLevel()
            end
        end
    end
end

function game.resetPlayer(x, y)
    x = x or game.getWidth() / 2 + 3
    y = y or 0
    console:write("Player position set to " .. x .. ", " .. y)
    player:reset(x, y)
end


function game.playSound(sound)
    assets.sfx[sound]:stop()
    assets.sfx[sound]:play()
end


function game.loadLevel(n)
    -- Cleanup state
    if game.currentPlanet then game.currentPlanet:halt() end
    dialog:clear()
    player.dead = false

    -- Get level data
    game.currentLevel = levels[n]
    console:write("Loading map " .. game.currentLevel.planet_def.map)

    -- Load planet data, hopefully this doesnt take too long
    game.currentPlanet = game.currentLevel:loadPlanet()

    -- Setup gravity controller
    game.currentPlanet:setGravityController(game.currentLevel.gravity or {})

    -- Parse initial dialog and run 0.5 seconds into level
    time.after(0.25, function()
        for i, message in ipairs(game.currentLevel:getDialog()) do
            dialog:push(message)
        end
    end)
end

function game.completeLevel()
    if game.completeLevelLock then return end
    game.completeLevelLock = true
    time.after(1, function() 
        game.completeLevelLock = false
    end)
    game.activeShader = assets.shaders.blur
    game.shaderMod = 0.0
    game.overlay = 0.0
    player.dead = true
    tween(0.3, game, { shaderMod = -3.0, overlay = 1, }, 'linear', function()
        game.overlay = 0.0
        game.activeShader = nil
        game.nextLevel()
        player.dead = false
    end)
end

function game.nextLevel()
    game.levelCount = game.levelCount + 1
    if game.levelCount == #levels then
        assets.music.music:stop()
        game.completed = true
        game.hideClock = true
    end
    if game.levelCount > #levels then
        return game.win()
    end
    return game.loadLevel(game.levelCount)
end

function game.prevLevel()
    game.levelCount = game.levelCount - 1
    if game.levelCount < 0 then
        return game.exit()
    end
    return game.loadLevel(game.levelCount)
end

function game.reload()
    game.loadLevel(game.levelCount)
end

function game.exit()
    require 'os'
    os.exit()
end

function game.win()
    game.exit()
end

function game.lose()
    game.completed = true
    game.hideMenu = true
    game.timeleft = 0
    game.currentPlanet:halt()     
    game.playSound('die')
    local x, y = game.toScreen(player:cx(), player:cy())
    local system = particles:enable('death', x, y)
    player.dead = true
    time.after(2, function()
        dialog:push("SCIENTIST: It seems as though our fate has been determined...", game.exit)
    end)
end 

function game.die()
    game.currentPlanet:halt()     
    game.playSound('die')
    console:write("Player died at " .. player.x .. ", " .. player.y)     
    local x, y = game.toScreen(player:cx(), player:cy())
    local system = particles:enable('death', x, y)
    player.dead = true
    time.after(0.75, function()
        particles:disable('death')
        game.activeMenu = game.deathMenu
    end) 
end

function game.checkCollision(entity)
    if player.dead or game.INVINCIBLE then
        return
    end
    if entity:getCollisionRect():intersects(player:getCollisionRect()) then
        if entity.isEnemy then
            game.die()
        end
    end
    local triggerRect = entity:getTriggerRect()
    if triggerRect and triggerRect:intersects(player:getCollisionRect()) then
        entity:trigger()
    end
end

function game.planetRotated(n)
    -- Chuck player if hes on ground...
    if player.state == PlayerState.GROUND and not game.INVINCIBLE then
        player:chuck(n)
    end
end


function game.fireBullet(x, y, left)
    game.playSound('laser')
    local bullet = Bullet()
    bullet.x = x
    bullet.y = y
    bullet.speed = left and -Bullet.SPEED or Bullet.SPEED
    game.currentPlanet.entities:insert(bullet)
end
