--[[

#########################################################################
#                                                                       #
# camera.lua                                                            #
#                                                                       #
# Simple Love2D camera class                                            #
#                                                                       #
# Copyright 2011 Josh Bothun                                            #
# joshbothun@gmail.com                                                  #
# http://minornine.com                                                  #
#                                                                       #
# This program is free software: you can redistribute it and/or modify  #
# it under the terms of the GNU General Public License as published by  #
# the Free Software Foundation, either version 3 of the License, or     #
# (at your option) any later version.                                   #
#                                                                       #
# This program is distributed in the hope that it will be useful,       #
# but WITHOUT ANY WARRANTY; without even the implied warranty of        #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
# GNU General Public License <http://www.gnu.org/licenses/> for         #
# more details.                                                         #
#                                                                       #
#########################################################################

--]]

require 'leaf.object'
require 'leaf.vector'
require 'leaf.rect'

local Camera = leaf.Object:extend()

function Camera:init(track_func)
    if track_func then self:track(track_func) end
    self.scale = 1
end

-- Set the cameras tracking function
function Camera:track(track_func)
    assert(type(track_func) == 'function')
    self.track_func = track_func
end

-- Return the upper left corner of the camera in world space
function Camera:getPosition()
    local tx, ty = 0, 0
    if self.track_func then
        tx, ty = self.track_func()
    end
    return tx - love.graphics.getWidth() / 2 / self.scale, ty - love.graphics.getHeight() / 2 / self.scale
end

-- Return a leaf.Rect representing the viewable rectangle in world space
function Camera:getClip()
    local x, y = self:getPosition()
    return leaf.Rect(x, y, x + love.graphics.getWidth(), y + love.graphics.getHeight())
end

-- Sets up matrix to center the active target
-- If a Z parameter is specified, it is considered a depth factor relative to the target
-- e.g., if z = 2.0, objects will appear 2x as close as the target
function Camera:push(z)  
    -- Default depth to 1, which is the plane of the target
    local z = z or 1
    local x, y = self:getPosition()

    -- Use builtin matrix
    love.graphics.push()

    -- Center on target, offset depth by Z, scale by camera scale
    love.graphics.scale(self.scale, self.scale)
    love.graphics.translate(z * -x, z * -y)
end

function Camera:pop()
    love.graphics.pop()
end

-- Convert a vector in screen space to world space.
-- ("World space" means the coordinate space of the camera's target)
function Camera:toWorld(x, y)
    local cam_x, cam_y = self:getPosition()
    if isinstance(x, leaf.Vector) then
        return x:translated(cam_x, cam_y)
    else
        return x + cam_x, y + cam_y
    end
end

-- Convert a vector in world space to screen space.
-- ("World space" means the coordinate space of the camera's target)
function Camera:toScreen(x, y)
    local cam_x, cam_y = self:getPosition()
    if isinstance(x, leaf.Vector) then
        return x:translated(-cam_x, -cam_y)
    else
        return x - cam_x, y - cam_y
    end
end


-- Namespace exports
leaf.Camera = Camera
