require 'math'
require 'entities'
require 'utils'

local ROTATE_SPEED = 0.3
local PING_DELAY = 1.5

DIRMAP = {
    [0] = 'down',
    [1] = 'left',
    [2] = 'up',
    [3] = 'right',
}

Tile = Object:extend()

Tile.Type = {
    SOLID='SOLID',
    AIR='AIR',
}

function Tile:init(rect, planet)
    self.planet = planet
    self.origin_rect = rect:copy()
    self.rect = rect
    self.type = Tile.Type.AIR
    self.intersected = false
    self.orientation = 0

    -- Quad needs to be set by planet
    self.quad = nil
end

function Tile:draw()
    local xfix = self.orientation >= 1 and self.orientation <= 2 and -TILE_SIZE or 0
    local yfix = self.orientation >= 2 and self.orientation <= 3 and -TILE_SIZE or 0

    love.graphics.push()
        local rect = self.planet:toOrigin(self.rect)
        love.graphics.translate(rect.left - xfix, rect.top - yfix)
        love.graphics.rotate(math.pi * self.orientation / 2)
        love.graphics.setColor(255,255,255)
        if game.COLLISION and self.intersected then
            love.graphics.setColor(200, 200, 200)
        end
        love.graphics.drawq(assets.gfx.planet_tileset, self.quad, 0, 0)
    love.graphics.pop()

end


Cloud = Object:extend()

function Cloud:init(img)
    self.x = game.getWidth()
    self.y = math.random(1, game.getHeight() - 10)
    self.speed = math.random(-30, -2)
    self.img = img
end

function Cloud:update(dt)
    self.x = self.x + self.speed * dt
end

function Cloud:draw()
    love.graphics.setColor(255, 255, 255, 200)
    love.graphics.draw(self.img, self.x, self.y)
end


Planet = Object:extend()

PLANET_SIZE = 240
TILE_COUNT = 12
TILE_SIZE = PLANET_SIZE / TILE_COUNT

function getLayer(map, name)
    for i, tilelayer in ipairs(map.layers) do
        if tilelayer.name == name then
            return tilelayer
        end
    end
    return {}
end

function Planet:toWorld(a, b)
    if not b then
        return a:translated(self.rect.left + PLANET_SIZE / 2, self.rect.top + PLANET_SIZE / 2)
    else
        return a + self.rect.left + PLANET_SIZE / 2, b + self.rect.top + PLANET_SIZE / 2
    end
end

function Planet:toOrigin(a, b)
    if not b then
        return a:translated(-self.rect.left - PLANET_SIZE / 2, -self.rect.top - PLANET_SIZE / 2)
    else
        return a - self.rect.left - PLANET_SIZE / 2, b - self.rect.top - PLANET_SIZE / 2
    end
end

function Planet:realignTiles()
    local angle = self.orientation * math.pi / 2
    local sin_theta = math.sin(angle)
    local cos_theta = math.cos(angle)
    for i = 1, #self.tiles do
        for j = 1, #self.tiles[i] do
            -- Put rect in correct origin space with 0,0 at center
            local temp_rect = self:toOrigin(self.tiles[i][j].origin_rect)

            -- Apply rotation
            local x = temp_rect.left * cos_theta - temp_rect.top * sin_theta
            local y = temp_rect.left * sin_theta + temp_rect.top * cos_theta
            temp_rect:moveTo(x, y)

            -- Transform real rectangle
            self.tiles[i][j].rect = self:toWorld(temp_rect)
            self.tiles[i][j].orientation = self.orientation

            local xfix = self.orientation >= 1 and self.orientation <= 2 and -TILE_SIZE or 0
            local yfix = self.orientation >= 2 and self.orientation <= 3 and -TILE_SIZE or 0
            self.tiles[i][j].rect:translate(xfix, yfix)
        end
    end
end

function Planet:realignEntities(numRotations)
    local angle = -math.pi * numRotations / 2
    local sin_theta = math.sin(-angle)
    local cos_theta = math.cos(-angle)
    for entity in self.entities:iter() do
        -- Put  in correct origin space with 0,0 at center
        local ox, oy = self:toOrigin(entity.x, entity.y)
        local rx = ox * cos_theta - oy * sin_theta
        local ry = ox * sin_theta + oy * cos_theta

        local xfix, yfix = 0, 0
        if numRotations > 0 then
            xfix = -entity.w
        else
            yfix = -entity.h
        end

        -- Transform real entity
        local wx, wy = self:toWorld(rx, ry)
        entity.x, entity.y = wx + xfix, wy + yfix
        entity:rotate(numRotations)
    end
end



function Planet:init(opt)
    opt = opt or {}
    for k, v in pairs(opt) do
        self[k] = v
    end

    -- Position planet in center of screen
    self.rect = Rect(PLANET_SIZE, PLANET_SIZE)
    local x = (game.getWidth() - self.rect:width()) / 2
    local y = (game.getHeight() - self.rect:height()) / 2
    self.rect:moveTo(x, y)

    -- Construct tiles
    self.tiles = {}
    local tw = self.rect:w() / TILE_COUNT
    local th = self.rect:h() / TILE_COUNT
    for i =1, TILE_COUNT do
        self.tiles[i] = {}
        for j = 1, TILE_COUNT do
            local t_rect = Rect(tw, th)
            t_rect:moveTo(self.rect.left + (i-1) * tw, self.rect.top + (j-1) * th)
            self.tiles[i][j] = Tile(t_rect, self)
        end
    end

    self.tiles[3][1].type = Tile.Type.AIR
    self.tiles[3][2].type = Tile.Type.AIR

    -- Direction planet is facing, 0/1/2/3
    self.orientation = 0
    self.fakeRotation = 0

    -- Load map data
    self.map = assets.maps[self.map]()
    self:loadTiles()

    -- Bg effects
    self.clouds = List()
    time.after(1, function() self:newCloud() end)

    -- Other crap
    self.tween = {}
    self.timer = {}

    -- Parse object tiles
    self.entities = List()
    self:parseEntities()
end

function Planet:setGravityController(controller)
    self.gravityController = controller
    if controller.time then
        self.timer.gravity = math.random(controller.time[1], controller.time[2])
    end
end


function Planet:parseEntities()
    local objlayer = getLayer(self.map, 'obj')
    for i, obj_def in ipairs(objlayer.objects) do
        local ref_or_class = entitymap[obj_def.gid]
        local worldx = obj_def.x + self.rect.left
        local worldy = obj_def.y + self.rect.top

        -- Fix position from import
        obj_def.properties.orientation = tonumber(obj_def.properties.o or obj_def.properties.orientation)
        if obj_def.properties.left and obj_def.properties.left == 'false' then
            obj_def.properties.left = false
        end
        worldy = worldy - TILE_SIZE
        if ref_or_class then
            if ref_or_class == 'player' then
                game.resetPlayer(worldx + 8, worldy - 4)
            else
                local entity = ref_or_class(obj_def.properties)
                entity.x, entity.y = worldx, worldy
                entity:adjust()
                self.entities:insert(entity)
            end
        end
    end
    -- for i = 1, #self.tiles do
    --     for j = 1, #self.tiles[i] do
    --         local idx = i + (j - 1) * TILE_COUNT
    --         local t_rect = self.tiles[i][j].rect
    --         local object_ref = objlayer.data[idx]
    --         if object_ref > 0 then
    --             local ref_or_class = entitymap[object_ref]
    --             if ref_or_class then
    --                 if ref_or_class == 'player' then
    --                     game.resetPlayer(t_rect.left + 10, t_rect.top)
    --                 else
    --                     local entity = ref_or_class()
    --                     entity.x, entity.y = t_rect.left, t_rect.top
    --                     self.entities:insert(entity)
    --                 end
    --             end
    --         end
    --     end
    -- end
end


function Planet:loadTiles()
    local bglayer = getLayer(self.map, 'bg')
    local clayer = getLayer(self.map, 'c')
    for i = 1, #self.tiles do
        for j = 1, #self.tiles[i] do
            -- Create quad for tile object
            local idx = i + (j - 1) * TILE_COUNT
            local tileset_ref = bglayer.data[idx]
            local tileset_x = ((tileset_ref - 1) % TILE_COUNT) * TILE_SIZE
            local tileset_y = (math.ceil(tileset_ref / TILE_COUNT) - 1) * TILE_SIZE
            self.tiles[i][j].quad = 
                love.graphics.newQuad(tileset_x, tileset_y,
                                      TILE_SIZE, TILE_SIZE, 
                                      PLANET_SIZE, PLANET_SIZE)

            -- Set collision data
            if clayer.data[idx] == 144 then
                self.tiles[i][j].type = Tile.Type.SOLID
            end
        end
    end
end

function Planet:tileContaining(x, y)
    for i = 1, #self.tiles do
        for j = 1, #self.tiles[i] do
            local tile = self.tiles[i][j]
            if tile.rect:contains(x, y) then
                return tile
            end
        end
    end
    return nil
end

function Planet:getSolidTileDir(x, y, dir)
    if dir == 'down' then
        y = y + TILE_SIZE
    elseif dir == 'left' then
        x = x - TILE_SIZE
    elseif dir == 'up' then
        y = y - TILE_SIZE
    elseif dir == 'right' then
        x = x + TILE_SIZE
    end
    local tile = self:tileContaining(x, y)
    local rect = tile and tile.rect or nil
    if tile and tile.type == Tile.Type.SOLID then
        return tile, rect
    end
    return nil, rect
end

function Planet:itemAt(x, y)
    local rect = Rect(x, y, x+player.w, y+player.h)
    for entity in self.entities:iter() do
        if entity.isItem then
            local erect = Rect(entity.x, entity.y, entity.x+entity.w, entity.y+entity.h)
            erect:centerScale(0.5)
            if rect:intersects(erect) then
                return entity
            end
        end
    end
    return nil
end

function Planet:newCloud()
    local image = assets.gfx['cloud' + math.random(1, 3)]
    self.clouds:insert(Cloud(image))
    time.after(math.random(4, 15), function() self:newCloud() end)
end

function Planet:updateEffects(dt)
    for cloud in self.clouds:iter() do
        cloud:update(dt)
    end
end

function Planet:triggerGravity()
    self.timer.gravity = math.random(self.gravityController.time[1],
                                      self.gravityController.time[2])
    local dir = self.gravityController.dir or (math.random() > 0.5 and 1 or -1)
    local x, y = game.toScreen(self:gravityRodPosition())
    if dir > 0 then
        particles:enable('ping_right', x, y)
        game.playSound('ping_right')
    else
        particles:enable('ping_left', x, y)
        game.playSound('ping_left')
    end
    self.gravityTimer = time.after(self.gravityController and self.gravityController.delay or PING_DELAY, function()
        self:rotatePlanet(dir)
    end)
end

function Planet:halt()
    self.halted = true
    if self.gravityTimer then
        self.gravityTimer:kill()
        self.gravityTimer = nil
    end
end

function Planet:gravityRodPosition()
    for entity in self.entities:iter() do
        if isinstance(entity, GravityRod) then
            return entity:cx(), entity:cy()
        end
    end
    return player.x, player.y
end

function Planet:update(dt)
    if self.halted then return end

    if self.timer.gravity then
        self.timer.gravity = self.timer.gravity - dt
        if self.timer.gravity < 0 then
            self:triggerGravity()
        end
    end

    for entity in self.entities:iter() do
        if entity.x < 0 or entity.x > game.getWidth() or entity.y < 0 or entity.y > game.getHeight() then
            self.entities:remove(entity)
        else
            entity:update(dt)
            game.checkCollision(entity)
        end
    end
end

function Planet:drawBg()
    -- Draw bg
    love.graphics.setColor(unpack(self.bgcolor))
    love.graphics.rectangle('fill', 0, 0, game.getWidth(), game.getHeight())
end


function Planet:draw()
    -- Draw clouds
    for cloud in self.clouds:iter() do
        cloud:draw()
    end

    -- Draw planet tiles with orientation
    love.graphics.push()
        local rect = self:toWorld(Rect())
        love.graphics.translate(rect.left, rect.top)
        love.graphics.rotate(self.fakeRotation)
        for i = 1, #self.tiles do
            for j = 1, #self.tiles[i] do
                self.tiles[i][j]:draw()
            end
        end

        -- Draw entities
        for entity in self.entities:iter() do
            local screen_x, screen_y = self:toOrigin(entity.x, entity.y)
            love.graphics.push()
                love.graphics.translate(screen_x, screen_y)
                    love.graphics.translate(entity.w / 2, entity.h / 2)
                    love.graphics.rotate(math.pi * entity.orientation / 2)
                    if entity.left then
                        love.graphics.scale(-1, 1)
                    end
                    love.graphics.translate(-entity.w / 2, -entity.h / 2)
                entity:draw()
            love.graphics.pop()
        end
    love.graphics.pop()

    if game.COLLISION then
        -- Draw absolute bb positions
        love.graphics.setColor(0, 255, 255)
        love.graphics.setLineWidth(1)
        love.graphics.rectangle('line', player:getCollisionRect():unpack())
        for entity in self.entities:iter() do
            love.graphics.rectangle('line', entity:getCollisionRect():unpack())
            -- local triggerRect = entity:getTriggerRect()
            -- if triggerRect then 
            --     love.graphics.setColor(0, 255, 128)
            --     love.graphics.rectangle('line', triggerRect:unpack())
            -- end
        end
    end
end

function Planet:rotatePlanet(n)
    game.playSound('rotate')
    self.orientation = (self.orientation + n) % 4
    if self.orientation < 0 then self.orientation = 3 end
    local angle = -math.pi * n / 2
    self:queueRotAnim(angle)
    self:realignTiles(n)
    self:realignEntities(n)

    game.planetRotated(n)
end

-- Fake rotation animation
function Planet:queueRotAnim(from)
    if self.tween.fakeRotation then
        tween.reset(self.tween.fakeRotation)
    end
    self.fakeRotation = from
    self.tween.fakeRotation = tween(ROTATE_SPEED, self, { fakeRotation = 0}, 'outCubic')
end
