require 'entity'
require 'entities'
require 'animation'
require 'math'

local JUMP_ACC = 4200
local JUMP_ENERGY = 0.06
local MAX_VELX = 30
local ACCEL = 1500
local FRICTION = 0.7
local GRAVITY = 900
local CHUCK_Y = {300, 500}
local CHUCK_X = {100, 600}

local STEP_SPEED = 0.1

player = Entity {
    w = 10,
    h = 12,
    left = false,
}

local State = {
    AIR='AIR',
    GROUND='GROUND',
}
PlayerState = State

function player:chuck(n)
    self.vely = -math.random(unpack(CHUCK_Y))
    self.velx = math.random(unpack(CHUCK_X))
    if n < 0 then
        self.velx = -self.velx
    end
    self.chucking = true
    self.state = State.AIR
end


function player:fixCollision(rect)
    local sx = 0.75
    rect.left = rect.left + (1 - sx) * rect:w() / 2
    rect.right = rect.right - (1 - sx) * rect:w() / 2
    return rect
end

function player:init()
    self.dead = false
    self.chucking = false
    self.state = State.AIR
    self.sprites = {
        idle = Animation(assets.gfx.guy_idle, 10, 0.1),
        jump = Animation(assets.gfx.guy_run, 10, 0.02),
        run = Animation(assets.gfx.guy_run, 10),
    }
    self.sprite = self.sprites.run
    self.jumpEnergy = 0

    -- Timer stuff
    self.steptimer = 0

    -- Setup friend
    self.y_history = Queue()
    self.friend = Friend()
    self:resetFriend()
end


function player:reset(x, y)
    self.x = x
    self.y = y
    self.velx = 0
    self.vely = 0
    self.accx = 0
    self.accy = 0
    self:resetFriend()
end

function player:resetFriend()
    self.friend.x = self.x - 10
    self.friend.y = self.y
end

function player:draw()
    love.graphics.setColor(255, 255, 255)
    if not self.dead then
        self.sprite:draw()
    end
end


function player:parseInput(dt)
    local k = love.keyboard.isDown
    if k('a') or k('left') then
        self.left = true
        self.accx = -ACCEL
        if self.state ~= State.AIR then
            if self.steptimer < 0 then
                game.playSound('step')
                self.steptimer = STEP_SPEED
            end
            self.sprite = self.sprites.run
        end
        if self.chucking then
            self.accx = self.accx * 0.3
        end
    elseif k('d') or k('right') then
        self.left = false
        self.accx = ACCEL
        if self.state ~= State.AIR then
            self.sprite = self.sprites.run
            if self.steptimer < 0 then
                game.playSound('step')
                self.steptimer = STEP_SPEED
            end
        end
        if self.chucking then
            self.accx = self.accx * 0.3
        end
    else
        if state ~= State.AIR then
            self.sprite = self.sprites.idle
        end
        self.accx = 0
    end



    if k('w') or k('up') or k('j') then
        self:jump(dt)
    else
        self.jumpEnergy = 0
    end
end

function player:useItem()
    -- See if theres an object at my location
    local item = game.currentPlanet:itemAt(self.x, self.y)
    if item then
        local ret = item:activate()
    end
end


function player:jump(dt)
    if self.state ~= State.AIR then
        game.playSound('jump')
        self.jumpEnergy = JUMP_ENERGY
        self.state = State.AIR
        self.sprite = self.sprites.jump
    end
    if self.jumpEnergy > 0 then
        self.jumpEnergy = self.jumpEnergy - dt
        self.vely = self.vely - JUMP_ACC * dt
    end
end


function player:update(dt)
    if self.dead then return end

    self.steptimer = self.steptimer - dt

    self:parseInput(dt)

    -- Apply friction
    if not self.chucking then
        self.velx = self.velx * FRICTION
    end
    if math.abs(self.velx) < 0.01 then
        self.velx = 0
    end

    -- Apply gravity
    self.accy = GRAVITY

    -- Lookahead position and see if we collide with tiles
    local nx, ny = self:stepPhysics(dt)
    local next_rect = Rect(nx, ny, nx+self.w, ny+self.h)
    local tiles = game.currentPlanet.tiles
    for i = 1, #tiles do
        for j = 1, #tiles[i] do
            local t_rect = tiles[i][j].rect
            if t_rect:intersects(next_rect) and tiles[i][j].type == Tile.Type.SOLID then
                tiles[i][j].intersected = true
                if nx + self.w > t_rect.left and nx < t_rect.right then
                    if self.y + self.h <= t_rect.top and ny + self.h >= t_rect.top then
                        ny = t_rect.top - self.h
                        self.vely = 0
                        self.state = State.GROUND
                        self.chucking = false
                    elseif self.y >= t_rect.bottom and ny <= t_rect.bottom then
                        ny = t_rect.bottom
                        self.vely = 0
                        self.chucking = false
                    end
                end
                if ny + self.h > t_rect.top and ny < t_rect.bottom then
                    if self.x + self.w <= t_rect.left and nx + self.w >= t_rect.left then
                        nx = t_rect.left - self.w
                        self.velx = 0
                        self.chucking = false
                    elseif self.x >= t_rect.right and nx <= t_rect.right then
                        nx = t_rect.right
                        self.velx = 0
                        self.chucking = false
                    end
                end
            else
                tiles[i][j].intersected = false
            end
        end
    end
    
    if ny ~= self.y then
        self.y_history:push(ny)
    end
    self.x, self.y = nx, ny
end


function player:keypressed(key, unicode)
    if key == 's' or key == 'down' then
        self:useItem()
    end
end
